
import os
import subprocess
import sys
import shutil

# Define directories
base_dir = os.path.abspath("sue_core")
modules_dir = os.path.join(base_dir, "sue_modules")
logs_dir = os.path.join(base_dir, "sue_logs")
rss_module_file = os.path.join(modules_dir, "sue_rss_feed_parser.py")

# Required packages
required_packages = ["feedparser", "beautifulsoup4", "requests"]

def install_packages():
    for package in required_packages:
        subprocess.check_call([sys.executable, "-m", "pip", "install", package])

def create_directories():
    os.makedirs(modules_dir, exist_ok=True)
    os.makedirs(logs_dir, exist_ok=True)

def write_rss_module():
    rss_script = """import feedparser
import requests
from bs4 import BeautifulSoup
from datetime import datetime
import hashlib
import os
import json

rss_feeds = [
    "http://feeds.reuters.com/reuters/topNews",
    "https://www.zerohedge.com/rss.xml",
    "https://rss.nytimes.com/services/xml/rss/nyt/Technology.xml",
    "https://www.aljazeera.com/xml/rss/all.xml",
    "https://feeds.bbci.co.uk/news/world/rss.xml"
]

output_folder = os.path.join("sue_core", "sue_modules", "sue_news_index")
os.makedirs(output_folder, exist_ok=True)

def clean_html(raw_html):
    soup = BeautifulSoup(raw_html, "html.parser")
    return soup.get_text()

def generate_hash(text):
    return hashlib.sha256(text.encode('utf-8')).hexdigest()

def summarize(text, max_words=60):
    words = text.split()
    return " ".join(words[:max_words]) + ("..." if len(words) > max_words else "")

indexed_articles = []
for url in rss_feeds:
    feed = feedparser.parse(url)
    for entry in feed.entries[:5]:
        title = entry.get("title", "")
        link = entry.get("link", "")
        summary_raw = entry.get("summary", "")
        published = entry.get("published", datetime.utcnow().isoformat())

        clean_summary = clean_html(summary_raw)
        short_summary = summarize(clean_summary)

        article_id = generate_hash(title + link)
        index_entry = {
            "id": article_id,
            "title": title,
            "summary": short_summary,
            "source": feed.feed.get("title", "Unknown Source"),
            "link": link,
            "published": published,
            "topics": []
        }

        indexed_articles.append(index_entry)

        with open(os.path.join(output_folder, f"{article_id}.json"), "w") as f:
            json.dump(index_entry, f, indent=2)

print(f"Saved {len(indexed_articles)} articles to '{output_folder}' folder.")
"""
    with open(rss_module_file, "w") as f:
        f.write(rss_script)

def main():
    print("Installing SUE Base Environment...")
    install_packages()
    create_directories()
    write_rss_module()
    print("SUE Base Installation Complete.")
    print(f"RSS Module installed to: {rss_module_file}")

if __name__ == "__main__":
    main()
