import os
import json
import time
import uuid
from pathlib import Path

try:
    import chromadb
    from chromadb.config import Settings
except ImportError as e:
    print("ChromaDB not available. Install with: pip install chromadb")
    raise

HOME = str(Path.home())
BASE = os.path.join(HOME, ".sue")
DB_DIR = os.path.join(BASE, "memory", "vector_db")
PROFILE_FILE = os.path.join(BASE, "profile.json")

os.makedirs(DB_DIR, exist_ok=True)
os.makedirs(os.path.join(BASE, "memory"), exist_ok=True)

def get_or_create(client, name):
    """Get existing collection or create new one if it doesn't exist."""
    names = [c.name for c in client.list_collections()]
    return client.get_collection(name) if name in names else client.create_collection(name)

# 1) Ensure profile.json
profile = {"name": "Josh", "aliases": ["Rave"], "created": time.time()}
if os.path.exists(PROFILE_FILE):
    try:
        with open(PROFILE_FILE, "r") as f:
            existing = json.load(f)
        # Update with new values only if key doesn't exist or is empty
        existing.update({k: v for k, v in profile.items() if k not in existing or not existing[k]})
        profile = existing
    except (json.JSONDecodeError, IOError) as e:
        print(f"Warning: Could not read existing profile file: {e}")
        # Continue with default profile

# Write updated profile
try:
    with open(PROFILE_FILE, "w") as f:
        json.dump(profile, f, indent=2)
except IOError as e:
    print(f"Error: Could not write profile file: {e}")
    raise

# 2) Seed profile facts
try:
    client = chromadb.Client(Settings(persist_directory=DB_DIR, anonymized_telemetry=False))
    profile_col = get_or_create(client, "sue_profile")
except Exception as e:
    print(f"Error: Could not initialize ChromaDB client: {e}")
    raise

facts = [
    "User prefers chillstep in the morning and black coffee.",
    "Morning routine: coffee, joint, and meds, then headlines.",
    "Alias: Rave.",
    "Josh appreciates a Matrix-themed neon UI.",
    "Daily need: medicine reminder in the morning.",
    "News preference: concise headline summary first, deep dive on request.",
    "Project focus: SUE local assistant, Anonymous.click build; MAC 5000 docs are active but lower priority until legal matters are handled.",
    "Tone preference: casual, playful, flirty, but precise and non-hallucinated answers.",
    "OSINT/cybersec enthusiast; enjoys automation and scripts.",
]

metas = [{
    "type": "profile",
    "subject": profile.get("name", "user"),
    "importance": "medium",
    "ts": time.time()
} for _ in facts]

ids = [str(uuid.uuid4()) for _ in facts]

# Add facts to collection
try:
    profile_col.add(documents=facts, metadatas=metas, ids=ids)
    client.persist()
    print("Seeded core profile for:", profile.get("name"))
    print("Aliases:", ", ".join(profile.get("aliases", [])))
    print("Facts added:", len(facts))
    print("Done.")
except Exception as e:
    print(f"Error: Could not add facts to database: {e}")
    raise