import os, json, time, uuid
from pathlib import Path

try:
    import chromadb
    from chromadb.config import Settings
except Exception as e:
    print("ChromaDB not available. Install with: pip install chromadb")
    raise

HOME = str(Path.home())
BASE = os.path.join(HOME, ".sue")
DB_DIR = os.path.join(BASE, "memory", "vector_db")
PROFILE_FILE = os.path.join(BASE, "profile.json")

os.makedirs(DB_DIR, exist_ok=True)
os.makedirs(os.path.join(BASE, "memory"), exist_ok=True)

def get_or_create(client, name):
    names = [c.name for c in client.list_collections()]
    return client.get_collection(name) if name in names else client.create_collection(name)

# 1) Ensure profile.json
profile = {"name": "Josh", "aliases": ["Rave"], "created": time.time()}
if os.path.exists(PROFILE_FILE):
    try:
        with open(PROFILE_FILE, "r") as f:
            existing = json.load(f)
        existing.update({k: v for k, v in profile.items() if k not in existing or not existing[k]})
        profile = existing
    except Exception:
        pass

with open(PROFILE_FILE, "w") as f:
    json.dump(profile, f, indent=2)

# 2) Seed profile facts
client = chromadb.Client(Settings(persist_directory=DB_DIR, anonymized_telemetry=False))
profile_col = get_or_create(client, "sue_profile")

facts = [
    "User prefers chillstep in the morning and black coffee.",
    "Morning routine: coffee, joint, and meds, then headlines.",
    "Alias: Rave.",
    "Josh appreciates a Matrix-themed neon UI.",
    "Daily need: medicine reminder in the morning.",
    "News preference: concise headline summary first, deep dive on request.",
    "Project focus: SUE local assistant, Anonymous.click build; MAC 5000 docs are active but lower priority until legal matters are handled.",
    "Tone preference: casual, playful, flirty, but precise and non-hallucinated answers.",
    "OSINT/cybersec enthusiast; enjoys automation and scripts.",
]

metas = [{
    "type": "profile",
    "subject": profile.get("name", "user"),
    "importance": "medium",
    "ts": time.time()
} for _ in facts]

ids = [str(uuid.uuid4()) for _ in facts]

# Add facts (de-dup lightly by trying to query first)
profile_col.add(documents=facts, metadatas=metas, ids=ids)
client.persist()

print("Seeded core profile for:", profile.get("name"))
print("Aliases:", ", ".join(profile.get("aliases", [])))
print("Facts added:", len(facts))
print("Done.")