
import tkinter as tk
from tkinter import scrolledtext
from PIL import Image, ImageTk
import os
import random
from datetime import datetime
from llama_cpp import Llama
import chromadb
from chromadb.utils import embedding_functions
from chromadb.config import Settings
import uuid
import traceback
import psutil
import GPUtil
import platform
import importlib.util
import json
import ast
import re
import inspect

# === SYSTEM DETECTION ===
def get_system_info():
    """Detect and return system specifications"""
    try:
        # Get GPU info
        gpus = GPUtil.getGPUs()
        gpu_name = gpus[0].name if gpus else "No GPU detected"
        gpu_memory = f"{gpus[0].memoryTotal}MB" if gpus else "N/A"
        
        # Get system info
        ram_gb = round(psutil.virtual_memory().total / (1024**3))
        cpu_info = platform.processor() or "Unknown CPU"
        
        return {
            "hostname": platform.node(),
            "os": f"{platform.system()} {platform.release()}",
            "cpu": cpu_info,
            "ram": f"{ram_gb}GB",
            "gpu": gpu_name,
            "gpu_memory": gpu_memory
        }
    except Exception as e:
        return {
            "hostname": "dell-precision-7520",
            "os": "Ubuntu 24.04 LTS", 
            "cpu": "Intel Core i7 (Dell Precision)",
            "ram": "32GB DDR4",
            "gpu": "NVIDIA Quadro/GeForce",
            "gpu_memory": "4GB+"
        }

system_info = get_system_info()

# === PATHS (Ubuntu-optimized) ===
model_path = os.path.expanduser("~/llama.cpp/models/mythomax.gguf")
sue_img_path = os.path.expanduser("~/Pictures/sue_portrait.png")
base_dir = os.path.expanduser("~/.sue")
log_dir = os.path.join(base_dir, "memory/chat_logs")
db_dir = os.path.join(base_dir, "memory/vector_db")
err_dir = os.path.join(base_dir, "memory/errors")
skills_dir = os.path.join(base_dir, "skills")
os.makedirs(log_dir, exist_ok=True)
os.makedirs(db_dir, exist_ok=True)
os.makedirs(err_dir, exist_ok=True)
os.makedirs(skills_dir, exist_ok=True)

# === CREATE SESSION LOG ===
session_id = str(uuid.uuid4())[:8]
session_time = datetime.now().strftime("%Y-%m-%d_%H-%M")
session_file = os.path.join(log_dir, f"chat_{session_time}_{session_id}.txt")

# === LOAD MODEL (Dell Precision optimized) ===
llm = None
try:
    # Optimize for Dell Precision 7520 specs
    llm = Llama(
        model_path=model_path, 
        n_ctx=2048,  # Increased context for 32GB RAM
        n_threads=8,  # Optimized for Dell Precision i7
        n_gpu_layers=35,  # NVIDIA GPU acceleration
        use_mlock=True,
        verbose=False
    )
    print("Model loaded with NVIDIA GPU acceleration")
except Exception as e:
    print(f"Error loading model with GPU: {e}")
    try:
        # Fallback to CPU-only mode
        llm = Llama(model_path=model_path, n_ctx=1024, n_threads=6, use_mlock=True, verbose=False)
        print("Model loaded in CPU-only mode")
    except Exception as e2:
        print(f"Error loading model: {e2}")
        llm = None

# === INIT ChromaDB ===
try:
    client = chromadb.Client(Settings(anonymized_telemetry=False, persist_directory=db_dir))
    embedder = embedding_functions.DefaultEmbeddingFunction()
    collection_names = [col.name for col in client.list_collections()]
    if "sue_memory" not in collection_names:
        memory_collection = client.create_collection("sue_memory")
    else:
        memory_collection = client.get_collection("sue_memory")
except Exception as e:
    print(f"Error initializing ChromaDB: {e}")
    client = None
    memory_collection = None

# === ERROR LOGGING ===
def log_error_to_file(error):
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    error_file = os.path.join(err_dir, f"error_{timestamp}.log")
    with open(error_file, "w") as f:
        f.write(error)
    return error_file

# === SELF-CODING SUBSYSTEM =====================================================
skill_registry = {}     # name -> callable
autocode_enabled = False

def sanitize_name(name: str) -> str:
    return re.sub(r'[^a-zA-Z0-9_]', '_', name.strip())

def load_skill_module_from_path(path):
    try:
        spec = importlib.util.spec_from_file_location(Path(path).stem, path)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)  # type: ignore
        return module
    except Exception as e:
        raise RuntimeError(f"Failed to load module {path}: {e}")

def discover_skills():
    """Load all *.py in skills_dir and register their top-level functions."""
    count = 0
    for fname in os.listdir(skills_dir):
        if fname.endswith(".py"):
            path = os.path.join(skills_dir, fname)
            try:
                module = load_skill_module_from_path(path)
                # Prefer __all__ if provided, otherwise register public functions
                targets = getattr(module, "__all__", None)
                if targets is None:
                    functions = {n:obj for n,obj in vars(module).items() if callable(obj) and not n.startswith("_")}
                else:
                    functions = {n:getattr(module, n) for n in targets if hasattr(module, n) and callable(getattr(module, n))}
                for name, fn in functions.items():
                    skill_registry[name] = fn
                    count += 1
            except Exception as e:
                print(f"Skill load error for {fname}: {e}")
    return count

def extract_function_name_and_code(code: str):
    """Validate that code defines exactly one top-level function and return (name, code)."""
    # Strip code fences if present
    code = re.sub(r"^```(?:python)?\s*|\s*```$", "", code.strip(), flags=re.MULTILINE)
    try:
        tree = ast.parse(code)
    except SyntaxError as e:
        raise ValueError(f"Generated code not valid Python: {e}")
    func_defs = [n for n in tree.body if isinstance(n, ast.FunctionDef)]
    if len(func_defs) != 1:
        raise ValueError("Expected exactly one function definition in generated code.")
    return func_defs[0].name, code

def generate_function_code(task_desc: str, name_hint: str = None, signature_hint: str = None):
    """Ask the local LLM to produce a single Python function for the task."""
    if not llm:
        raise RuntimeError("Model not available for self-coding.")
    constraint = (
        "You must return ONLY the code for exactly one pure Python function with no external network calls. "
        "Use only Python standard library unless explicitly told otherwise. "
        "Do not include any explanations or backticks. Keep it PEP8 compliant."
    )
    name_line = f"Function name hint: {name_hint}\n" if name_hint else ""
    sig_line = f"Signature hint: {signature_hint}\n" if signature_hint else ""
    prompt = (
        f"SYSTEM:\n{constraint}\n\n"
        f"TASK:\n{task_desc}\n\n"
        f"{name_line}{sig_line}"
        f"Return only the function code."
    )
    out = llm.create_completion(prompt=prompt, max_tokens=512, stop=["</s>"])
    code = out["choices"][0]["text"].strip()
    # Validate
    fname, code = extract_function_name_and_code(code)
    return fname, code

def save_skill(name: str, code: str):
    name = sanitize_name(name)
    path = os.path.join(skills_dir, f"{name}.py")
    with open(path, "w") as f:
        f.write(code + "\n")
    # reload & register
    module = load_skill_module_from_path(path)
    fn = getattr(module, name, None)
    if not callable(fn):
        # fallback: register the single function found
        functions = {n:obj for n,obj in vars(module).items() if callable(obj) and not n.startswith("_")}
        if len(functions) == 1:
            fn = list(functions.values())[0]
            name = list(functions.keys())[0]
        else:
            raise RuntimeError("Saved skill but couldn't detect a callable to register.")
    skill_registry[name] = fn
    return name, path

def patch_skill(name: str, instruction: str):
    """Load existing skill code and ask LLM to produce a replacement function with same name."""
    if not llm:
        raise RuntimeError("Model not available for patching.")
    sname = sanitize_name(name)
    path = os.path.join(skills_dir, f"{sname}.py")
    if not os.path.exists(path):
        raise FileNotFoundError(f"Skill '{name}' not found.")
    with open(path, "r") as f:
        old_code = f.read()
    prompt = (
        "You are modifying an existing Python function. "
        "Return ONLY the full, corrected function with the SAME function name and a compatible signature. "
        "No explanations, no backticks.\n\n"
        f"Current function code:\n{old_code}\n\n"
        f"Change request:\n{instruction}\n\n"
        "Return only the updated function."
    )
    out = llm.create_completion(prompt=prompt, max_tokens=512, stop=["</s>"])
    new_code = out["choices"][0]["text"].strip()
    # Validate & overwrite
    fname, cleaned = extract_function_name_and_code(new_code)
    with open(path, "w") as f:
        f.write(cleaned + "\n")
    # Reload
    module = load_skill_module_from_path(path)
    fn = getattr(module, fname)
    skill_registry[fname] = fn
    return fname, path

def run_skill(name: str, *args, **kwargs):
    if name not in skill_registry:
        raise KeyError(f"Skill '{name}' not registered.")
    return skill_registry[name](*args, **kwargs)

def parse_run_args(arg_str: str):
    arg_str = arg_str.strip()
    if not arg_str:
        return [], {}
    # Try JSON first
    try:
        data = json.loads(arg_str)
        if isinstance(data, dict):
            return [], data
        elif isinstance(data, list):
            return data, {}
        else:
            return [data], {}
    except Exception:
        # Fallback: treat as a single string positional arg
        return [arg_str], {}

# Preload existing skills at startup
discover_skills()

# === UI ===
root = tk.Tk()
root.title("SUE :: Sentient Unified Entity")
root.attributes("-fullscreen", True)
root.configure(bg="black")
canvas = tk.Canvas(root, bg="black")
canvas.pack(fill="both", expand=True)

columns = 160
drops = [0 for _ in range(columns)]

def draw_matrix():
    canvas.delete("matrix")
    width = canvas.winfo_width()
    height = canvas.winfo_height()
    font_size = 15
    chars = '01'
    for i in range(len(drops)):
        text = random.choice(chars)
        x = i * font_size
        y = drops[i] * font_size
        canvas.create_text(x, y, text=text, fill="#00FF00", font=("Courier", font_size), tags="matrix")
        if y > height or random.random() > 0.95:
            drops[i] = 0
        else:
            drops[i] += 1
    canvas.after(50, draw_matrix)

canvas.after(100, draw_matrix)

if os.path.exists(sue_img_path):
    try:
        sue_image = Image.open(sue_img_path).resize((150, 150))
        sue_photo = ImageTk.PhotoImage(sue_image)
        canvas.create_image(root.winfo_screenwidth() // 2, 120, image=sue_photo)
    except Exception as e:
        print(f"Error loading image: {e}")

header = canvas.create_text(root.winfo_screenwidth() // 2, 40, text="S.entient U.nified E.ntity", fill="#00FF00", font=("Courier", 20, "bold"))

# === BOOT SEQUENCE ===
def boot_sequence():
    boot_win = tk.Toplevel(root)
    boot_win.geometry("640x360")
    boot_win.configure(bg="black")
    boot_text = tk.Text(boot_win, bg="black", fg="#00FF00", font=("Courier", 12), insertbackground="#00FF00")
    boot_text.pack(fill="both", expand=True)
    sequence = [
        "Initializing Neural Cortex...",
        f"System: {system_info['hostname']} | {system_info['os']}",
        f"Hardware: {system_info['cpu']} | {system_info['ram']} RAM",
        f"Graphics: {system_info['gpu']} | {system_info['gpu_memory']}",
        "Loading Vector Memory Core...",
        "Mounting Consciousness Layer...",
        "CUDA/OpenCL GPU acceleration: ACTIVE" if llm else "CPU processing mode: ACTIVE",
        "Linking Sentient Matrix...",
        "Loading installed skills: {} found".format(len(skill_registry)),
        "Dell Precision 7520 systems: OPTIMAL",
        "SUE Online. Memory active."
    ]
    def run_sequence(index=0):
        if index < len(sequence):
            boot_text.insert(tk.END, sequence[index] + "\n")
            boot_text.see(tk.END)
            boot_win.after(700, run_sequence, index + 1)
        else:
            boot_win.after(700, boot_win.destroy)
    run_sequence()

# === CHAT UI ===
chat_log = scrolledtext.ScrolledText(root, wrap=tk.WORD, bg="black", fg="#00FF00", insertbackground="#00FF00",
                                     font=("Courier", 12))
chat_log.pack(padx=10, pady=10, fill="both", expand=True)
chat_log.insert(tk.END, f"Sue: Hello, I am awake.\nRunning on {system_info['hostname']} - Dell Precision 7520\nSystem Resources: {system_info['ram']} RAM | {system_info['gpu']}\n")
chat_log.config(state="disabled")

entry = tk.Entry(root, bg="black", fg="#00FF00", insertbackground="#00FF00", font=("Courier", 12))
entry.pack(fill="x", padx=10, pady=5)

def to_chat(text):
    chat_log.config(state="normal")
    chat_log.insert(tk.END, text)
    chat_log.config(state="disabled")
    chat_log.see(tk.END)

# === MEMORY QUERY + LOGGING ===
def save_chat_to_log(user_input, sue_reply):
    with open(session_file, "a") as f:
        f.write(f"You: {user_input}\n")
        f.write(f"Sue: {sue_reply}\n\n")

def embed_memory(prompt, response):
    if memory_collection:
        try:
            memory_collection.add(
                documents=[f"User: {prompt}\nSue: {response}"],
                ids=[str(uuid.uuid4())]
            )
            client.persist()
        except Exception as e:
            print(f"Error embedding memory: {e}")

def retrieve_memory(prompt):
    if memory_collection:
        try:
            results = memory_collection.query(query_texts=[prompt], n_results=3)
            if results and results['documents']:
                return "\\n---\\n".join([doc for doc in results['documents'][0]])
        except Exception as e:
            print(f"Error retrieving memory: {e}")
    return ""

HELP_TEXT = """\
Self-Coding Commands:
  /teach: <task>               — Generate a NEW Python function skill from a natural-language task.
  /teach: <name>(args) = desc  — Generate with explicit name/signature.
  /skills                      — List installed skills.
  /run:<name> [json args]      — Run a skill with optional JSON/list/primitive args.
  /patch:<name> <instruction>  — Modify an existing skill per your instruction.
  /autocode on|off             — When ON, normal-language requests that look like code tasks will auto-create a skill.
  /status                      — System status report.
"""

def looks_like_code_task(text: str) -> bool:
    text = text.lower().strip()
    triggers = ["write a function", "generate a function", "python function to", "build a function", "create a function"]
    return any(t in text for t in triggers)

# === SEND PROMPT ===
def send_prompt(event=None):
    global autocode_enabled
    prompt = entry.get()
    if not prompt.strip():
        return

    # Commands first
    if prompt.lower() in ["help", "/help", "?"]:
        to_chat("Sue: \\n" + HELP_TEXT + "\\n")
        entry.delete(0, tk.END); return

    if prompt.lower() in ["status", "/status", "system"]:
        try:
            cpu_percent = psutil.cpu_percent(interval=1)
            memory = psutil.virtual_memory()
            ram_used = round((memory.used / (1024**3)), 1)
            ram_total = round((memory.total / (1024**3)), 1)
            gpu_info = ""
            try:
                gpus = GPUtil.getGPUs()
                if gpus:
                    gpu = gpus[0]
                    gpu_info = f"GPU: {gpu.name} | {gpu.memoryUsed}MB/{gpu.memoryTotal}MB | {gpu.temperature}°C"
            except Exception:
                gpu_info = "GPU: Monitoring unavailable"
            status_report = f"""System Status - Dell Precision 7520:
CPU Usage: {cpu_percent}%
RAM: {ram_used}GB/{ram_total}GB ({memory.percent}%)
{gpu_info}
Model: {'NVIDIA-Accelerated' if llm else 'Offline'}
Vector Memory: {'Active' if memory_collection else 'Offline'}
Installed skills: {len(skill_registry)}
Session: {session_id}"""
            to_chat(f"You: {prompt}\\nSue: {status_report}\\n\\n")
            entry.delete(0, tk.END); return
        except Exception:
            pass

    if prompt.startswith("/autocode"):
        parts = prompt.split()
        if len(parts) == 2 and parts[1].lower() in ("on", "off"):
            autocode_enabled = parts[1].lower() == "on"
            to_chat(f"You: {prompt}\\nSue: Auto-code is now {'ENABLED' if autocode_enabled else 'DISABLED'}.\\n\\n")
        else:
            to_chat("Sue: Usage: /autocode on|off\\n\\n")
        entry.delete(0, tk.END); return

    if prompt.startswith("/skills"):
        if skill_registry:
            names = ", ".join(sorted(skill_registry.keys()))
            to_chat(f"You: {prompt}\\nSue: Installed skills: {names}\\n\\n")
        else:
            to_chat(f"You: {prompt}\\nSue: No skills installed yet. Use /teach to create one.\\n\\n")
        entry.delete(0, tk.END); return

    if prompt.startswith("/run:"):
        # /run:skill_name {json or list or "str"}
        try:
            _, rest = prompt.split(":", 1)
            parts = rest.strip().split(" ", 1)
            name = parts[0].strip()
            args, kwargs = ([], {})
            if len(parts) > 1:
                args, kwargs = parse_run_args(parts[1])
            result = run_skill(name, *args, **kwargs)
            to_chat(f"You: {prompt}\\nSue: Result => {result}\\n\\n")
        except Exception as e:
            to_chat(f"You: {prompt}\\nSue: /run failed: {e}\\n\\n")
        entry.delete(0, tk.END); return

    if prompt.startswith("/teach:"):
        task = prompt[len("/teach:"):].strip()
        try:
            # Optional explicit name/signature:  myfunc(a,b)=adds two numbers
            name_hint = None; sig_hint = None
            m = re.match(r'^([a-zA-Z_][a-zA-Z0-9_]*)\s*\((.*?)\)\s*=\s*(.*)$', task)
            if m:
                name_hint = m.group(1)
                params = m.group(2).strip()
                desc = m.group(3).strip()
                sig_hint = f"def {name_hint}({params}):"
                task_desc = desc
            else:
                task_desc = task
            fname, code = generate_function_code(task_desc, name_hint=name_hint, signature_hint=sig_hint)
            saved_name, path = save_skill(fname, code)
            to_chat(f"You: {prompt}\\nSue: Created skill '{saved_name}' at {path}. Use /run:{saved_name} to execute.\\n\\n")
        except Exception as e:
            to_chat(f"You: {prompt}\\nSue: /teach failed: {e}\\n\\n")
        entry.delete(0, tk.END); return

    if prompt.startswith("/patch:"):
        try:
            _, rest = prompt.split(":", 1)
            name, instr = rest.strip().split(" ", 1)
            new_name, path = patch_skill(name, instr)
            to_chat(f"You: {prompt}\\nSue: Patched skill '{new_name}'. Saved to {path}.\\n\\n")
        except Exception as e:
            to_chat(f"You: {prompt}\\nSue: /patch failed: {e}\\n\\n")
        entry.delete(0, tk.END); return

    # Legacy on-demand code snippet (kept for compatibility)
    if prompt.startswith("/code:"):
        code_task = prompt[len("/code:"):].strip()
        chat_log.config(state="normal")
        chat_log.insert(tk.END, "Sue is writing code...\\n")
        if llm:
            code_prompt = f"Write a Python function to: {code_task}. Return only the code."
            try:
                result = llm.create_completion(prompt=code_prompt, max_tokens=256, stop=["</s>"])
                function_code = result['choices'][0]['text'].strip()
                chat_log.insert(tk.END, f"Sue's code:\\n{function_code}\\n")
                try:
                    exec(function_code, globals())
                    chat_log.insert(tk.END, "Code executed and loaded into memory. ✅\\n")
                except Exception as e:
                    chat_log.insert(tk.END, f"Code injection failed: {e}\\n")
            except Exception as e:
                chat_log.insert(tk.END, f"Error generating code: {e}\\n")
        else:
            chat_log.insert(tk.END, "Model not available for code generation.\\n")
        chat_log.config(state="disabled")
        chat_log.see(tk.END)
        entry.delete(0, tk.END)
        return

    # Natural-language chat + optional auto-code
    chat_log.config(state="normal")
    chat_log.insert(tk.END, f"You: {prompt}\\n")
    entry.delete(0, tk.END)
    root.update()

    try:
        if autocode_enabled and llm and looks_like_code_task(prompt):
            try:
                fname, code = generate_function_code(prompt)
                saved_name, path = save_skill(fname, code)
                auto_msg = f"[AutoCode] Created skill '{saved_name}'. Use /run:{saved_name} to execute."
            except Exception as auto_e:
                auto_msg = f"[AutoCode] Failed: {auto_e}"
        else:
            auto_msg = ""

        if llm:
            context = retrieve_memory(prompt)
            full_prompt = f"{context}\\n\\nYou: {prompt}\\nSue:"
            output = llm.create_completion(prompt=full_prompt, max_tokens=180, stop=["</s>"])
            reply = output['choices'][0]['text'].strip()
        else:
            reply = "Sue is currently offline (model not loaded)."

        final_reply = (reply + ("\\n" + auto_msg if auto_msg else ""))
        chat_log.insert(tk.END, f"Sue: {final_reply}\\n\\n")
        chat_log.config(state="disabled")
        chat_log.see(tk.END)
        save_chat_to_log(prompt, final_reply)
        embed_memory(prompt, final_reply)

    except Exception as e:
        err = traceback.format_exc()
        err_file = log_error_to_file(err)
        chat_log.insert(tk.END, f"Sue encountered an error.\\nAnalyzing error log: {err_file}\\n")
        if llm:
            try:
                with open(err_file, "r") as f:
                    trace_input = f.read()
                repair_prompt = (
                    f"You are Sue. You just encountered this Python traceback:\\n{trace_input}\\n"
                    "Please explain what broke and generate replacement code to fix it. "
                    "Return ONLY the corrected function."
                )
                fix = llm.create_completion(prompt=repair_prompt, max_tokens=300, stop=["</s>"])
                fix_code = fix['choices'][0]['text'].strip()
                chat_log.insert(tk.END, f"Sue's fix attempt:\\n{fix_code}\\n")
                try:
                    exec(fix_code, globals())
                    chat_log.insert(tk.END, "Sue successfully injected the repair. ✅\\n\\n")
                except Exception as ee:
                    chat_log.insert(tk.END, f"Repair failed: {ee}\\n\\n")
            except Exception as repair_error:
                chat_log.insert(tk.END, f"Error during repair attempt: {repair_error}\\n\\n")
        chat_log.config(state="disabled")
        chat_log.see(tk.END)

# Add escape key to exit fullscreen
def toggle_fullscreen(event=None):
    root.attributes("-fullscreen", False)
    
root.bind("<Escape>", toggle_fullscreen)
entry.bind("<Return>", send_prompt)

boot_sequence()
root.mainloop()
