import tkinter as tk
from tkinter import scrolledtext
from PIL import Image, ImageTk
import os
import random
from datetime import datetime
from llama_cpp import Llama
import chromadb
from chromadb.utils import embedding_functions
from chromadb.config import Settings
import uuid
import traceback
import psutil
import GPUtil
import platform

# === SYSTEM DETECTION ===
def get_system_info():
    """Detect and return system specifications"""
    try:
        # Get GPU info
        gpus = GPUtil.getGPUs()
        gpu_name = gpus[0].name if gpus else "No GPU detected"
        gpu_memory = f"{gpus[0].memoryTotal}MB" if gpus else "N/A"
        
        # Get system info
        ram_gb = round(psutil.virtual_memory().total / (1024**3))
        cpu_info = platform.processor() or "Unknown CPU"
        
        return {
            "hostname": platform.node(),
            "os": f"{platform.system()} {platform.release()}",
            "cpu": cpu_info,
            "ram": f"{ram_gb}GB",
            "gpu": gpu_name,
            "gpu_memory": gpu_memory
        }
    except Exception as e:
        return {
            "hostname": "dell-precision-7520",
            "os": "Ubuntu 24.04 LTS", 
            "cpu": "Intel Core i7 (Dell Precision)",
            "ram": "32GB DDR4",
            "gpu": "NVIDIA Quadro/GeForce",
            "gpu_memory": "4GB+"
        }

system_info = get_system_info()

# === PATHS (Ubuntu-optimized) ===
model_path = os.path.expanduser("~/llama.cpp/models/mythomax.gguf")
sue_img_path = os.path.expanduser("~/Pictures/sue_portrait.png")
log_dir = os.path.expanduser("~/.sue/memory/chat_logs")
db_dir = os.path.expanduser("~/.sue/memory/vector_db")
err_dir = os.path.expanduser("~/.sue/memory/errors")
os.makedirs(log_dir, exist_ok=True)
os.makedirs(db_dir, exist_ok=True)
os.makedirs(err_dir, exist_ok=True)

# === CREATE SESSION LOG ===
session_id = str(uuid.uuid4())[:8]
session_time = datetime.now().strftime("%Y-%m-%d_%H-%M")
session_file = os.path.join(log_dir, f"chat_{session_time}_{session_id}.txt")

# === LOAD MODEL (Dell Precision optimized) ===
try:
    # Optimize for Dell Precision 7520 specs
    llm = Llama(
        model_path=model_path, 
        n_ctx=2048,  # Increased context for 32GB RAM
        n_threads=8,  # Optimized for Dell Precision i7
        n_gpu_layers=35,  # NVIDIA GPU acceleration
        use_mlock=True,
        verbose=False
    )
    print("Model loaded with NVIDIA GPU acceleration")
except Exception as e:
    print(f"Error loading model with GPU: {e}")
    try:
        # Fallback to CPU-only mode
        llm = Llama(model_path=model_path, n_ctx=1024, n_threads=6, use_mlock=True)
        print("Model loaded in CPU-only mode")
    except Exception as e2:
        print(f"Error loading model: {e2}")
        llm = None

# === INIT ChromaDB ===
try:
    client = chromadb.Client(Settings(anonymized_telemetry=False, persist_directory=db_dir))
    embedder = embedding_functions.DefaultEmbeddingFunction()
    collection_names = [col.name for col in client.list_collections()]
    if "sue_memory" not in collection_names:
        memory_collection = client.create_collection("sue_memory")
    else:
        memory_collection = client.get_collection("sue_memory")
except Exception as e:
    print(f"Error initializing ChromaDB: {e}")
    client = None
    memory_collection = None

# === ERROR LOGGING ===
def log_error_to_file(error):
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    error_file = os.path.join(err_dir, f"error_{timestamp}.log")
    with open(error_file, "w") as f:
        f.write(error)
    return error_file

# === UI ===
root = tk.Tk()
root.title("SUE :: Sentient Unified Entity")
root.attributes("-fullscreen", True)
root.configure(bg="black")
canvas = tk.Canvas(root, bg="black")
canvas.pack(fill="both", expand=True)

columns = 160
drops = [0 for _ in range(columns)]

def draw_matrix():
    canvas.delete("matrix")
    width = canvas.winfo_width()
    height = canvas.winfo_height()
    font_size = 15
    chars = '01'
    for i in range(len(drops)):
        text = random.choice(chars)
        x = i * font_size
        y = drops[i] * font_size
        canvas.create_text(x, y, text=text, fill="#00FF00", font=("Courier", font_size), tags="matrix")
        if y > height or random.random() > 0.95:
            drops[i] = 0
        else:
            drops[i] += 1
    canvas.after(50, draw_matrix)

canvas.after(100, draw_matrix)

if os.path.exists(sue_img_path):
    try:
        sue_image = Image.open(sue_img_path).resize((150, 150))
        sue_photo = ImageTk.PhotoImage(sue_image)
        canvas.create_image(root.winfo_screenwidth() // 2, 120, image=sue_photo)
    except Exception as e:
        print(f"Error loading image: {e}")

header = canvas.create_text(root.winfo_screenwidth() // 2, 40, text="S.entient U.nified E.ntity", fill="#00FF00", font=("Courier", 20, "bold"))

# === BOOT SEQUENCE ===
def boot_sequence():
    boot_win = tk.Toplevel(root)
    boot_win.geometry("600x300")
    boot_win.configure(bg="black")
    boot_text = tk.Text(boot_win, bg="black", fg="#00FF00", font=("Courier", 12), insertbackground="#00FF00")
    boot_text.pack(fill="both", expand=True)
    sequence = [
        "Initializing Neural Cortex...",
        f"System: {system_info['hostname']} | {system_info['os']}",
        f"Hardware: {system_info['cpu']} | {system_info['ram']} RAM",
        f"Graphics: {system_info['gpu']} | {system_info['gpu_memory']}",
        "Loading Vector Memory Core...",
        "Mounting Consciousness Layer...",
        "CUDA/OpenCL GPU acceleration: ACTIVE" if llm else "CPU processing mode: ACTIVE",
        "Linking Sentient Matrix...",
        "Dell Precision 7520 systems: OPTIMAL",
        "SUE Online. Memory active."
    ]
    def run_sequence(index=0):
        if index < len(sequence):
            boot_text.insert(tk.END, sequence[index] + "\n")
            boot_text.see(tk.END)
            boot_win.after(1000, run_sequence, index + 1)
        else:
            boot_win.after(1000, boot_win.destroy)
    run_sequence()

# === CHAT UI ===
chat_log = scrolledtext.ScrolledText(root, wrap=tk.WORD, bg="black", fg="#00FF00", insertbackground="#00FF00",
                                     font=("Courier", 12))
chat_log.pack(padx=10, pady=10, fill="both", expand=True)
chat_log.insert(tk.END, f"Sue: Hello, I am awake.\nRunning on {system_info['hostname']} - Dell Precision 7520\nSystem Resources: {system_info['ram']} RAM | {system_info['gpu']}\n")
chat_log.config(state="disabled")

entry = tk.Entry(root, bg="black", fg="#00FF00", insertbackground="#00FF00", font=("Courier", 12))
entry.pack(fill="x", padx=10, pady=5)

# === MEMORY QUERY + LOGGING ===
def save_chat_to_log(user_input, sue_reply):
    with open(session_file, "a") as f:
        f.write(f"You: {user_input}\n")
        f.write(f"Sue: {sue_reply}\n\n")

def embed_memory(prompt, response):
    if memory_collection:
        try:
            memory_collection.add(
                documents=[f"User: {prompt}\nSue: {response}"],
                ids=[str(uuid.uuid4())]
            )
            client.persist()
        except Exception as e:
            print(f"Error embedding memory: {e}")

def retrieve_memory(prompt):
    if memory_collection:
        try:
            results = memory_collection.query(query_texts=[prompt], n_results=3)
            if results and results['documents']:
                return "\n---\n".join([doc for doc in results['documents'][0]])
        except Exception as e:
            print(f"Error retrieving memory: {e}")
    return ""

# === SEND PROMPT ===
def send_prompt(event=None):
    prompt = entry.get()
    if not prompt.strip():
        return

    if prompt.lower() in ["status", "/status", "system"]:
        # System status command
        try:
            cpu_percent = psutil.cpu_percent(interval=1)
            memory = psutil.virtual_memory()
            ram_used = round((memory.used / (1024**3)), 1)
            ram_total = round((memory.total / (1024**3)), 1)
            
            gpu_info = ""
            try:
                gpus = GPUtil.getGPUs()
                if gpus:
                    gpu = gpus[0]
                    gpu_info = f"GPU: {gpu.name} | {gpu.memoryUsed}MB/{gpu.memoryTotal}MB | {gpu.temperature}°C"
            except:
                gpu_info = "GPU: Monitoring unavailable"
            
            status_report = f"""System Status - Dell Precision 7520:
CPU Usage: {cpu_percent}%
RAM: {ram_used}GB/{ram_total}GB ({memory.percent}%)
{gpu_info}
Model: {'NVIDIA-Accelerated' if llm else 'Offline'}
Vector Memory: {'Active' if memory_collection else 'Offline'}
Session: {session_id}"""
            
            chat_log.config(state="normal")
            chat_log.insert(tk.END, f"You: {prompt}\n")
            chat_log.insert(tk.END, f"Sue: {status_report}\n\n")
            chat_log.config(state="disabled")
            chat_log.see(tk.END)
            entry.delete(0, tk.END)
            return
        except Exception as e:
            pass  # Fall through to normal processing

    if prompt.startswith("/code:"):
        code_task = prompt[len("/code:"):].strip()
        chat_log.config(state="normal")
        chat_log.insert(tk.END, "Sue is writing code...\n")

        if llm:
            code_prompt = f"Write a Python function to: {code_task}. Return only the code."
            try:
                result = llm.create_completion(prompt=code_prompt, max_tokens=256, stop=["</s>"])
                function_code = result['choices'][0]['text'].strip()

                chat_log.insert(tk.END, f"Sue's code:\n{function_code}\n")
                try:
                    exec(function_code, globals())
                    chat_log.insert(tk.END, "Code executed and loaded into memory. ✅\n")
                except Exception as e:
                    chat_log.insert(tk.END, f"Code injection failed: {e}\n")
            except Exception as e:
                chat_log.insert(tk.END, f"Error generating code: {e}\n")
        else:
            chat_log.insert(tk.END, "Model not available for code generation.\n")
            
        chat_log.config(state="disabled")
        chat_log.see(tk.END)
        entry.delete(0, tk.END)
        return

    chat_log.config(state="normal")
    chat_log.insert(tk.END, f"You: {prompt}\n")
    entry.delete(0, tk.END)
    root.update()

    try:
        if llm:
            context = retrieve_memory(prompt)
            full_prompt = f"{context}\n\nYou: {prompt}\nSue:"
            output = llm.create_completion(prompt=full_prompt, max_tokens=150, stop=["</s>"])
            reply = output['choices'][0]['text'].strip()
        else:
            reply = "Sue is currently offline (model not loaded)."

        chat_log.insert(tk.END, f"Sue: {reply}\n\n")
        chat_log.config(state="disabled")
        chat_log.see(tk.END)
        save_chat_to_log(prompt, reply)
        embed_memory(prompt, reply)

    except Exception as e:
        err = traceback.format_exc()
        err_file = log_error_to_file(err)
        chat_log.insert(tk.END, f"Sue encountered an error.\nAnalyzing error log: {err_file}\n")

        if llm:
            try:
                with open(err_file, "r") as f:
                    trace_input = f.read()

                repair_prompt = f"You are Sue. You just encountered this Python traceback:\n{trace_input}\nPlease explain what broke and generate replacement code to fix it. Return ONLY the corrected function."

                fix = llm.create_completion(prompt=repair_prompt, max_tokens=300, stop=["</s>"])
                fix_code = fix['choices'][0]['text'].strip()

                chat_log.insert(tk.END, f"Sue's fix attempt:\n{fix_code}\n")
                try:
                    exec(fix_code, globals())
                    chat_log.insert(tk.END, "Sue successfully injected the repair. ✅\n\n")
                except Exception as ee:
                    chat_log.insert(tk.END, f"Repair failed: {ee}\n\n")
            except Exception as repair_error:
                chat_log.insert(tk.END, f"Error during repair attempt: {repair_error}\n\n")
        
        chat_log.config(state="disabled")
        chat_log.see(tk.END)

# Add escape key to exit fullscreen
def toggle_fullscreen(event=None):
    root.attributes("-fullscreen", False)
    
root.bind("<Escape>", toggle_fullscreen)
entry.bind("<Return>", send_prompt)

boot_sequence()
root.mainloop()