import os
import json
import time
import uuid
import random
import traceback
from datetime import datetime
from pathlib import Path

# UI
import tkinter as tk
from tkinter import scrolledtext
from PIL import Image, ImageTk

# System / HW
import platform
import psutil
try:
    import GPUtil
except Exception:
    GPUtil = None

# LLM + Vector DB
try:
    from llama_cpp import Llama
except Exception:
    Llama = None

try:
    import chromadb
    from chromadb.utils import embedding_functions
    from chromadb.config import Settings
except Exception:
    chromadb = None
    embedding_functions = None
    Settings = None


# ==============================
# System Info / Detection
# ==============================

def get_system_info():
    try:
        gpu_name = "No GPU detected"
        gpu_memory = "N/A"
        if GPUtil:
            gpus = GPUtil.getGPUs()
            if gpus:
                gpu_name = gpus[0].name
                gpu_memory = f"{getattr(gpus[0], 'memoryTotal', 'N/A')}MB"
        ram_gb = round(psutil.virtual_memory().total / (1024**3))
        cpu_info = platform.processor() or "Unknown CPU"
        return {
            "hostname": platform.node(),
            "os": f"{platform.system()} {platform.release()}",
            "cpu": cpu_info,
            "ram": f"{ram_gb}GB",
            "gpu": gpu_name,
            "gpu_memory": gpu_memory
        }
    except Exception:
        # Fallback (safe defaults for Dell Precision 7520 per user's env)
        return {
            "hostname": "dell-precision-7520",
            "os": "Ubuntu 24.04 LTS",
            "cpu": "Intel Core i7 (Dell Precision)",
            "ram": "32GB DDR4",
            "gpu": "NVIDIA Quadro/GeForce",
            "gpu_memory": "4GB+"
        }

system_info = get_system_info()


# ==============================
# Paths / Directories
# ==============================

home = str(Path.home())
base_dir = os.path.join(home, ".sue")
log_dir = os.path.join(base_dir, "memory", "chat_logs")
db_dir = os.path.join(base_dir, "memory", "vector_db")
err_dir = os.path.join(base_dir, "memory", "errors")
img_path = os.path.join(home, "Pictures", "sue_portrait.png")
profile_file = os.path.join(base_dir, "profile.json")

for p in (log_dir, db_dir, err_dir, base_dir):
    os.makedirs(p, exist_ok=True)

# Model path (user's existing setup)
model_path = os.path.expanduser("~/llama.cpp/models/mythomax.gguf")


# ==============================
# Session Log
# ==============================

session_id = str(uuid.uuid4())[:8]
session_time = datetime.now().strftime("%Y-%m-%d_%H-%M")
session_file = os.path.join(log_dir, f"chat_{session_time}_{session_id}.txt")


# ==============================
# LLM Load
# ==============================

llm = None
if Llama is None:
    print("llama_cpp not available; running in offline mode.")
else:
    try:
        llm = Llama(
            model_path=model_path,
            n_ctx=2048,
            n_threads=max(4, os.cpu_count() or 4),
            # modest default; user can tune; GPU offload requires proper build
            n_gpu_layers=35,
            use_mlock=True,
            verbose=False
        )
        print("Model loaded (GPU-accelerated if supported).")
    except Exception as e:
        print(f"GPU load failed: {e}")
        try:
            llm = Llama(
                model_path=model_path,
                n_ctx=1024,
                n_threads=max(4, (os.cpu_count() or 6) - 2),
                use_mlock=True,
                verbose=False
            )
            print("Model loaded in CPU-only mode.")
        except Exception as e2:
            print(f"Model load failed: {e2}")
            llm = None


# ==============================
# Vector Memory (ChromaDB)
# ==============================

client = None
memory_collection = None
profile_collection = None

def _chromadb_get_or_create(_client, name):
    names = [c.name for c in _client.list_collections()]
    return _client.get_collection(name) if name in names else _client.create_collection(name)

if chromadb and Settings:
    try:
        client = chromadb.Client(Settings(anonymized_telemetry=False, persist_directory=db_dir))
        # default embedder; ok for local small use
        embedder = embedding_functions.DefaultEmbeddingFunction() if embedding_functions else None
        memory_collection = _chromadb_get_or_create(client, "sue_memory")   # chat memory
        profile_collection = _chromadb_get_or_create(client, "sue_profile") # identity/preferences
    except Exception as e:
        print(f"Chroma init error: {e}")
        client = None
        memory_collection = None
        profile_collection = None
else:
    print("ChromaDB not available; memory will be disabled.")


# ==============================
# Profile Persistence
# ==============================

def load_profile():
    data = {"name": None, "aliases": [], "created": time.time()}
    try:
        if os.path.exists(profile_file):
            with open(profile_file, "r") as f:
                data.update(json.load(f))
    except Exception:
        pass
    return data

def save_profile(p):
    try:
        with open(profile_file, "w") as f:
            json.dump(p, f, indent=2)
    except Exception as e:
        print(f"Profile save error: {e}")

profile = load_profile()
if not profile.get("name"):
    # Default to Josh (user)
    profile["name"] = "Josh"
    save_profile(profile)


# ==============================
# Error Logging
# ==============================

def log_error_to_file(error_text):
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    error_file = os.path.join(err_dir, f"error_{timestamp}.log")
    try:
        with open(error_file, "w") as f:
            f.write(error_text)
    except Exception:
        pass
    return error_file


# ==============================
# Vector Memory Helpers
# ==============================

def remember_profile_fact(fact_text, importance="low"):
    """Store a persistent profile fact about the user."""
    if not profile_collection:
        return None
    _id = str(uuid.uuid4())
    meta = {
        "type": "profile",
        "subject": profile.get("name") or "user",
        "importance": importance,
        "ts": time.time(),
    }
    try:
        profile_collection.add(documents=[fact_text], metadatas=[meta], ids=[_id])
        client.persist()
        return _id
    except Exception as e:
        print(f"Profile fact store error: {e}")
        return None

def list_profile_facts(limit=50):
    """Return list of (id, doc, meta)."""
    if not profile_collection:
        return []
    subject = profile.get("name") or "user"
    try:
        res = profile_collection.query(query_texts=[subject], n_results=limit)
        docs = res.get("documents", [[]])[0]
        ids = res.get("ids", [[]])[0]
        metas = res.get("metadatas", [[]])[0]
        return list(zip(ids, docs, metas))
    except Exception as e:
        print(f"List profile facts error: {e}")
        return []

def search_memories(q, k=5):
    """Semantic search across profile + chat memories."""
    results = []
    try:
        if profile_collection:
            r = profile_collection.query(query_texts=[q], n_results=k)
            for i in range(len(r.get("ids",[[]])[0])):
                results.append(("profile",
                                r["ids"][0][i],
                                r["documents"][0][i],
                                r["metadatas"][0][i]))
    except Exception as e:
        print(f"profile search error: {e}")
    try:
        if memory_collection:
            r = memory_collection.query(query_texts=[q], n_results=k)
            for i in range(len(r.get("ids",[[]])[0])):
                results.append(("chat",
                                r["ids"][0][i],
                                r["documents"][0][i],
                                r["metadatas"][0][i]))
    except Exception as e:
        print(f"chat search error: {e}")
    return results

def forget_memory(mem_id):
    """Delete by ID from either collection."""
    for col in (profile_collection, memory_collection):
        if not col:
            continue
        try:
            col.delete(ids=[mem_id])
            if client:
                client.persist()
            return True
        except Exception:
            pass
    return False

def embed_chat_memory(prompt, response):
    """Store a chat exchange with metadata."""
    if not memory_collection:
        return
    try:
        memory_collection.add(
            documents=[f"User: {prompt}\nSue: {response}"],
            metadatas=[{
                "type": "chat",
                "subject": profile.get("name") or "user",
                "ts": time.time()
            }],
            ids=[str(uuid.uuid4())]
        )
        client.persist()
    except Exception as e:
        print(f"Chat memory store error: {e}")

def retrieve_profile_context(k=5):
    """Pull the most relevant profile facts to prime replies."""
    if not profile_collection:
        return ""
    try:
        subject = profile.get("name") or "user"
        res = profile_collection.query(query_texts=[subject], n_results=k)
        docs = res.get("documents", [[]])[0]
        if not docs:
            return ""
        return "\n".join([f"[Profile] {d}" for d in docs[:k]])
    except Exception as e:
        print(f"profile ctx error: {e}")
        return ""

def retrieve_chat_context(prompt, k=3):
    """Retrieve relevant prior chat snippets."""
    if not memory_collection:
        return ""
    try:
        results = memory_collection.query(query_texts=[prompt], n_results=k)
        if results and results.get('documents', [[]])[0]:
            return "\n---\n".join([doc for doc in results['documents'][0]])
    except Exception as e:
        print(f"chat ctx error: {e}")
    return ""


# ==============================
# UI Setup
# ==============================

root = tk.Tk()
root.title("SUE :: Sentient Unified Entity")
root.attributes("-fullscreen", True)
root.configure(bg="black")

canvas = tk.Canvas(root, bg="black", highlightthickness=0)
canvas.pack(fill="both", expand=True)

columns = 160
drops = [0 for _ in range(columns)]

def draw_matrix():
    canvas.delete("matrix")
    width = canvas.winfo_width()
    height = canvas.winfo_height()
    font_size = 15
    chars = '01'
    for i in range(len(drops)):
        text = random.choice(chars)
        x = i * font_size
        y = drops[i] * font_size
        canvas.create_text(x, y, text=text, fill="#00FF00", font=("Courier", font_size), tags="matrix")
        if y > height or random.random() > 0.95:
            drops[i] = 0
        else:
            drops[i] += 1
    canvas.after(50, draw_matrix)

canvas.after(100, draw_matrix)

# Optional avatar
if os.path.exists(img_path):
    try:
        sue_image = Image.open(img_path).resize((150, 150))
        sue_photo = ImageTk.PhotoImage(sue_image)
        canvas.create_image(root.winfo_screenwidth() // 2, 120, image=sue_photo)
    except Exception as e:
        print(f"Avatar load error: {e}")

header = canvas.create_text(root.winfo_screenwidth() // 2, 40,
                            text="S.entient U.nified E.ntity",
                            fill="#00FF00", font=("Courier", 20, "bold"))

# Boot sequence window
def boot_sequence():
    boot_win = tk.Toplevel(root)
    boot_win.geometry("640x360+80+80")
    boot_win.configure(bg="black")
    boot_text = tk.Text(boot_win, bg="black", fg="#00FF00",
                        font=("Courier", 12), insertbackground="#00FF00")
    boot_text.pack(fill="both", expand=True)
    sequence = [
        "Initializing Neural Cortex...",
        f"System: {system_info['hostname']} | {system_info['os']}",
        f"Hardware: {system_info['cpu']} | {system_info['ram']} RAM",
        f"Graphics: {system_info['gpu']} | {system_info['gpu_memory']}",
        "Loading Vector Memory Core...",
        "Mounting Consciousness Layer...",
        "CUDA/OpenCL GPU acceleration: ACTIVE" if llm else "CPU processing mode: ACTIVE",
        "Linking Sentient Matrix...",
        "Dell Precision 7520 systems: OPTIMAL",
        "SUE Online. Memory active."
    ]
    def run_sequence(index=0):
        if index < len(sequence):
            boot_text.insert(tk.END, sequence[index] + "\n")
            boot_text.see(tk.END)
            boot_win.after(900, run_sequence, index + 1)
        else:
            boot_win.after(800, boot_win.destroy)
    run_sequence()

# Chat widgets
chat_log = scrolledtext.ScrolledText(root, wrap=tk.WORD, bg="black", fg="#00FF00",
                                     insertbackground="#00FF00", font=("Courier", 12), height=20)
chat_log.pack(padx=10, pady=10, fill="both", expand=True)

def to_chat(text):
    chat_log.config(state="normal")
    chat_log.insert(tk.END, text)
    chat_log.config(state="disabled")
    chat_log.see(tk.END)

to_chat(
    f"Sue: Hello {profile.get('name')}, I am awake.\n"
    f"Running on {system_info['hostname']} - Dell Precision 7520\n"
    f"System Resources: {system_info['ram']} RAM | {system_info['gpu']}\n\n"
)

entry = tk.Entry(root, bg="black", fg="#00FF00", insertbackground="#00FF00", font=("Courier", 12))
entry.pack(fill="x", padx=10, pady=5)

# ==============================
# Command Help
# ==============================

HELP_TEXT = """Available commands:
  /status                      — Show system status.
  /iam <name>                  — Set your name (persisted).
  /whoami                      — Show stored name and aliases.
  /remember <fact>             — Save a personal fact (profile memory).
  /mem.list [profile|chat]     — List known facts/notes.
  /mem.search <query>          — Semantic search across memories.
  /mem.forget <id>             — Delete a memory by ID.
  /code: <task>                — Generate and inject a Python function for the task.
  /help                        — Show this help.
"""


# ==============================
# Core Chat + Commands
# ==============================

def save_chat_to_log(user_input, sue_reply):
    try:
        with open(session_file, "a") as f:
            f.write(f"You: {user_input}\n")
            f.write(f"Sue: {sue_reply}\n\n")
    except Exception:
        pass

def system_status_report():
    try:
        cpu_percent = psutil.cpu_percent(interval=1)
        memory = psutil.virtual_memory()
        ram_used = round((memory.used / (1024**3)), 1)
        ram_total = round((memory.total / (1024**3)), 1)

        gpu_info = "GPU: Monitoring unavailable"
        if GPUtil:
            try:
                gpus = GPUtil.getGPUs()
                if gpus:
                    gpu = gpus[0]
                    gpu_info = f"GPU: {gpu.name} | {getattr(gpu, 'memoryUsed', 'NA')}MB/{getattr(gpu, 'memoryTotal', 'NA')}MB | {getattr(gpu, 'temperature', 'NA')}°C"
            except Exception:
                pass

        status_report = f"""System Status - Dell Precision 7520:
CPU Usage: {cpu_percent}%
RAM: {ram_used}GB/{ram_total}GB ({memory.percent}%)
{gpu_info}
Model: {'Available' if llm else 'Offline'}
Vector Memory: {'Active' if client else 'Offline'}
Session: {session_id}"""
        return status_report
    except Exception as e:
        return "Status unavailable."

def handle_commands(prompt):
    """Return True if handled, else False."""
    pl = prompt.strip()

    if pl in ("/help", "help", "?"):
        to_chat(f"You: {prompt}\nSue:\n{HELP_TEXT}\n")
        return True

    if pl in ("status", "/status", "system"):
        report = system_status_report()
        to_chat(f"You: {prompt}\nSue: {report}\n\n")
        return True

    if pl.startswith("/iam "):
        _, nm = pl.split(" ", 1)
        profile["name"] = nm.strip()
        save_profile(profile)
        to_chat(f"You: {prompt}\nSue: Got it. I’ll call you {profile['name']} from now on.\n\n")
        return True

    if pl == "/whoami":
        aliases = ", ".join(profile.get("aliases", [])) or "—"
        to_chat(f"You: {prompt}\nSue: You are {profile.get('name')}. Aliases: {aliases}\n\n")
        return True

    if pl.startswith("/remember "):
        fact = pl[len("/remember "):].strip()
        _id = remember_profile_fact(fact)
        msg = f"Saved: { _id }" if _id else "Memory store offline."
        to_chat(f"You: {prompt}\nSue: {msg}\n\n")
        return True

    if pl.startswith("/mem.list"):
        parts = pl.split()
        which = parts[1].lower() if len(parts) > 1 else "profile"
        if which == "profile":
            items = list_profile_facts(limit=50)
            if not items:
                to_chat(f"You: {prompt}\nSue: No profile facts stored yet.\n\n")
            else:
                lines = [f"{i+1}. id={id_} :: {doc}" for i,(id_,doc,meta) in enumerate(items)]
                to_chat(f"You: {prompt}\nSue:\n" + "\n".join(lines) + "\n\n")
        else:
            subject = profile.get("name") or "user"
            hits = search_memories(subject, k=10)
            lines = [f"{i+1}. [{src}] id={id_} :: {doc[:120]}..." for i,(src,id_,doc,meta) in enumerate(hits)]
            to_chat(f"You: {prompt}\nSue:\n" + ("\n".join(lines) if lines else "No chat snippets found.") + "\n\n")
        return True

    if pl.startswith("/mem.search "):
        q = pl[len("/mem.search "):].strip()
        hits = search_memories(q, k=8)
        if not hits:
            to_chat(f"You: {prompt}\nSue: No matches.\n\n")
        else:
            lines = [f"{i+1}. [{src}] id={id_} :: {doc[:140]}..." for i,(src,id_,doc,meta) in enumerate(hits)]
            to_chat(f"You: {prompt}\nSue:\n" + "\n".join(lines) + "\n\n")
        return True

    if pl.startswith("/mem.forget "):
        mem_id = pl[len("/mem.forget "):].strip()
        ok = forget_memory(mem_id)
        to_chat(f"You: {prompt}\nSue: {'Deleted.' if ok else 'Could not delete (id not found?).'}\n\n")
        return True

    if pl.startswith("/code:"):
        code_task = pl[len("/code:"):].strip()
        to_chat("Sue is writing code...\n")
        if llm:
            code_prompt = f"Write a Python function to: {code_task}. Return only the code."
            try:
                result = llm.create_completion(prompt=code_prompt, max_tokens=256, stop=[\"</s>\"])
                function_code = result['choices'][0]['text'].strip()
                to_chat(f\"Sue's code:\\n{function_code}\\n\")
                try:
                    exec(function_code, globals())
                    to_chat("Code executed and loaded into memory. ✅\n\n")
                except Exception as e:
                    to_chat(f\"Code injection failed: {e}\\n\\n\")
            except Exception as e:
                to_chat(f\"Error generating code: {e}\\n\\n\")
        else:
            to_chat("Model not available for code generation.\n\n")
        return True

    return False

def send_prompt(event=None):
    prompt = entry.get()
    if not prompt.strip():
        return

    # Commands
    if handle_commands(prompt):
        entry.delete(0, tk.END)
        return

    # Normal chat
    to_chat(f"You: {prompt}\n")
    entry.delete(0, tk.END)
    root.update()

    try:
        if llm:
            profile_ctx = retrieve_profile_context(k=5)
            chat_ctx = retrieve_chat_context(prompt, k=3)
            context = (profile_ctx + ("\n---\n" if profile_ctx and chat_ctx else "") + chat_ctx).strip()
            full_prompt = ((f"{context}\n\n") if context else "") + f"You ({profile.get('name')}): {prompt}\nSue:"
            output = llm.create_completion(prompt=full_prompt, max_tokens=180, stop=[\"</s>\"])
            reply = output['choices'][0]['text'].strip()
        else:
            reply = "Sue is currently offline (model not loaded)."

        to_chat(f"Sue: {reply}\n\n")
        save_chat_to_log(prompt, reply)
        embed_chat_memory(prompt, reply)

    except Exception as e:
        err = traceback.format_exc()
        err_file = log_error_to_file(err)
        to_chat(f"Sue encountered an error.\nAnalyzing error log: {err_file}\n")

        if llm:
            try:
                with open(err_file, "r") as f:
                    trace_input = f.read()
                repair_prompt = (
                    "You are Sue. You just encountered this Python traceback:\n"
                    f"{trace_input}\n"
                    "Please explain what broke and generate replacement code to fix it. Return ONLY the corrected function."
                )
                fix = llm.create_completion(prompt=repair_prompt, max_tokens=320, stop=[\"</s>\"])
                fix_code = fix['choices'][0]['text'].strip()
                to_chat(f\"Sue's fix attempt:\\n{fix_code}\\n\")
                try:
                    exec(fix_code, globals())
                    to_chat("Sue successfully injected the repair. ✅\n\n")
                except Exception as ee:
                    to_chat(f\"Repair failed: {ee}\\n\\n\")
            except Exception as repair_error:
                to_chat(f\"Error during repair attempt: {repair_error}\\n\\n\")


# Key bindings
def toggle_fullscreen(event=None):
    root.attributes("-fullscreen", False)

root.bind("<Escape>", toggle_fullscreen)
entry.bind("<Return>", send_prompt)

boot_sequence()
root.mainloop()
